<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Student;
use App\Models\Course;
use App\Models\StudentCourse;
use App\Models\Center;
use App\Models\PaymentRecord;
use App\Models\ActivityLog;
use App\Models\Partner;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inertia\Inertia;

class AdminController extends Controller
{


    public function index(Request $request)
    {
        $query = Student::with(['courses', 'center']);
        
        // Apply filters
        if ($request->center_id) {
            $query->where('center_id', $request->center_id);
        }
        
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('state_code', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%');
            });
        }
        
        $students = $query->latest()->paginate(10)->withQueryString();
        
        // Time range filtering
        $timeFilter = $request->time_filter ?? 'current_month';
        $customMonth = $request->custom_month;
        $customYear = $request->custom_year ?? now()->year;
        
        [$startDate, $endDate, $timeLabel] = $this->getTimeRange($timeFilter, $customMonth, $customYear);
        
        $centerFilter = $request->center_id;
        
        // Apply center filter to metrics
        $studentQuery = Student::query();
        $enrollmentQuery = StudentCourse::query();
        
        if ($centerFilter) {
            $studentQuery->where('center_id', $centerFilter);
            $enrollmentQuery->whereHas('student', function($q) use ($centerFilter) {
                $q->where('center_id', $centerFilter);
            });
        }
        
        // Create separate queries for each metric to avoid conflicts
        $baseStudentQuery = Student::query();
        $baseEnrollmentQuery = StudentCourse::query();
        
        if ($centerFilter) {
            $baseStudentQuery->where('center_id', $centerFilter);
            $baseEnrollmentQuery->whereHas('student', function($q) use ($centerFilter) {
                $q->where('center_id', $centerFilter);
            });
        }
        
        $metrics = [
            'period_students' => (clone $baseStudentQuery)->whereBetween('students.created_at', [$startDate, $endDate])->count(),
            'total_students' => (clone $baseStudentQuery)->count(),
            'total_courses' => Course::when($centerFilter, function($q) use ($centerFilter) {
                return $q->where('center_id', $centerFilter);
            })->count(),
            'period_enrollments' => (clone $baseEnrollmentQuery)->whereBetween('student_courses.created_at', [$startDate, $endDate])->count(),
            'total_enrollments' => (clone $baseEnrollmentQuery)->count(),
            'period_certificates' => (clone $baseEnrollmentQuery)->where('student_courses.is_certified', true)
                                                   ->whereBetween('student_courses.certification_date', [$startDate, $endDate])
                                                   ->count(),
            'total_certificates' => (clone $baseEnrollmentQuery)->where('student_courses.is_certified', true)->count(),
            'pending_payments' => (clone $baseEnrollmentQuery)->where('student_courses.payment_status', 'pending')->count(),
            'period_payments' => (clone $baseEnrollmentQuery)->where('student_courses.payment_status', 'paid')
                                               ->whereBetween('student_courses.updated_at', [$startDate, $endDate])
                                               ->count(),
            'total_payments' => (clone $baseEnrollmentQuery)->where('student_courses.payment_status', 'paid')->count(),
            'outstanding_amount' => (clone $baseEnrollmentQuery)->join('courses', 'student_courses.course_id', '=', 'courses.id')
                                                  ->selectRaw('SUM(CASE WHEN (courses.price - COALESCE(student_courses.amount_paid, 0)) > 0 THEN (courses.price - COALESCE(student_courses.amount_paid, 0)) ELSE 0 END) as outstanding')
                                                  ->value('outstanding') ?: 0,
        ];
        
        $centers = Center::where('is_active', true)->get();
        
        return Inertia::render('Admin/Dashboard', [
            'students' => $students,
            'metrics' => $metrics,
            'centers' => $centers,
            'timeLabel' => $timeLabel,
            'filters' => $request->only(['center_id', 'search', 'time_filter', 'custom_month', 'custom_year'])
        ]);
    }

    public function create()
    {
        $courses = Course::where('is_active', true)->get();
        $centers = Center::where('is_active', true)->get();
        return Inertia::render('Admin/CreateStudent', [
            'courses' => $courses,
            'centers' => $centers
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'state_code' => 'required|string|unique:students,state_code|max:20',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'center_id' => 'required|exists:centers,id',
            'enrollment_date' => 'required|date',
            'courses' => 'required|array|min:1',
            'courses.*.course_id' => 'required|exists:courses,id',
            'courses.*.payment_status' => 'required|in:pending,paid,overdue',
        ]);

        $center = Center::find($validated['center_id']);
        
        $student = Student::create([
            'name' => $validated['name'],
            'state_code' => $validated['state_code'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'center' => $center->name,
            'center_id' => $validated['center_id'],
            'enrollment_date' => $validated['enrollment_date'],
            'created_by_type' => 'admin',
            'created_by_id' => Auth::id(),
            'created_by_name' => Auth::user()->name,
        ]);

        foreach ($validated['courses'] as $courseData) {
            StudentCourse::create([
                'student_id' => $student->id,
                'course_id' => $courseData['course_id'],
                'payment_status' => $courseData['payment_status'],
                'enrollment_date' => $validated['enrollment_date'],
            ]);
        }

        return redirect()->route('admin.index')->with('success', 'Student enrolled successfully!');
    }

    public function show(Student $student)
    {
        $student->load('courses');
        return Inertia::render('Admin/ShowStudent', [
            'student' => $student
        ]);
    }

    public function edit(Student $student)
    {
        $student->load('courses');
        $courses = Course::where('is_active', true)->get();
        return Inertia::render('Admin/EditStudent', [
            'student' => $student,
            'courses' => $courses
        ]);
    }

    public function update(Request $request, Student $student)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'state_code' => 'required|string|max:20|unique:students,state_code,' . $student->id,
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'center' => 'required|string|max:255',
            'enrollment_date' => 'required|date',
        ]);

        $student->update($validated);

        return redirect()->route('admin.index')->with('success', 'Student updated successfully!');
    }

    public function destroy(Student $student)
    {
        $student->delete();
        return redirect()->route('admin.index')->with('success', 'Student deleted successfully!');
    }

    // Course Management
    public function courses(Request $request)
    {
        $query = Course::query();
        
        if ($request->search) {
            $query->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
        }
        
        $courses = $query->latest()->paginate(15)->withQueryString();
        
        return Inertia::render('Admin/Courses', [
            'courses' => $courses,
            'filters' => $request->only(['search'])
        ]);
    }

    public function createCourse()
    {
        return Inertia::render('Admin/CreateCourse');
    }

    public function storeCourse(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|unique:courses,name|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration_months' => 'required|integer|min:1',
        ]);

        Course::create($validated);

        return redirect()->route('admin.courses')->with('success', 'Course created successfully!');
    }

    public function editCourse(Course $course)
    {
        return Inertia::render('Admin/EditCourse', [
            'course' => $course
        ]);
    }

    public function updateCourse(Request $request, Course $course)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:courses,name,' . $course->id,
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration_months' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);

        $course->update($validated);

        return redirect()->route('admin.courses')->with('success', 'Course updated successfully!');
    }

    public function destroyCourse(Course $course)
    {
        $course->delete();
        return redirect()->route('admin.courses')->with('success', 'Course deleted successfully!');
    }

    // Partner Management
    public function partners()
    {
        $partners = Partner::with('center')
            ->latest()
            ->paginate(10)
            ->through(function ($partner) {
                return [
                    'id' => $partner->id,
                    'name' => $partner->name,
                    'email' => $partner->email,
                    'access_code' => $partner->access_code,
                    'phone' => $partner->phone,
                    'is_active' => $partner->is_active,
                    'is_manager' => $partner->is_manager,
                    'center' => $partner->center,
                    'created_at' => $partner->created_at,
                    'updated_at' => $partner->updated_at,
                ];
            });
        
        return Inertia::render('Admin/Partners', [
            'partners' => $partners
        ]);
    }

    public function createPartner()
    {
        $centers = Center::where('is_active', true)->get();
        return Inertia::render('Admin/CreatePartner', [
            'centers' => $centers
        ]);
    }

    public function storePartner(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:partners,email',
            'center_id' => 'required|exists:centers,id',
            'phone' => 'nullable|string|max:20',
        ]);

        $validated['access_code'] = 'PART-' . strtoupper(Str::random(8));

        Partner::create($validated);

        return redirect()->route('admin.partners')->with('success', 'Partner created successfully!');
    }

    public function destroyPartner(Partner $partner)
    {
        $partner->delete();
        return redirect()->route('admin.partners')->with('success', 'Partner deleted successfully!');
    }
    
    public function resetPartnerCode(Partner $partner)
    {
        $partner->update([
            'access_code' => 'PART-' . strtoupper(Str::random(8))
        ]);
        
        return back()->with('success', 'Partner access code reset successfully!');
    }
    
    // Center Management
    public function centers()
    {
        $centers = Center::withCount(['students', 'partners'])->with('manager')->latest()->paginate(10);
        return Inertia::render('Admin/Centers', [
            'centers' => $centers
        ]);
    }
    
    public function createCenter()
    {
        $availablePartners = Partner::whereNull('center_id')->orWhere('is_manager', false)->get();
        return Inertia::render('Admin/CreateCenter', [
            'availablePartners' => $availablePartners
        ]);
    }
    
    public function storeCenter(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|unique:centers,name|max:255',
            'location' => 'required|string|max:255',
            'manager_id' => 'nullable|exists:partners,id',
        ]);

        $center = Center::create([
            'name' => $validated['name'],
            'location' => $validated['location'],
            'manager_name' => '',
            'contact_email' => '',
            'contact_phone' => '',
        ]);
        
        if ($validated['manager_id']) {
            $partner = Partner::find($validated['manager_id']);
            $partner->update([
                'center_id' => $center->id,
                'is_manager' => true
            ]);
            
            $center->update([
                'manager_name' => $partner->name,
                'contact_email' => $partner->email,
                'contact_phone' => $partner->phone ?? '',
            ]);
        }
        
        return redirect()->route('admin.centers')->with('success', 'Center created successfully!');
    }
    
    public function editCenter(Center $center)
    {
        $center->load('manager');
        $availablePartners = Partner::where(function($q) use ($center) {
            $q->whereNull('center_id')
              ->orWhere('center_id', $center->id)
              ->orWhere('is_manager', false);
        })->get();
        
        return Inertia::render('Admin/EditCenter', [
            'center' => $center,
            'availablePartners' => $availablePartners
        ]);
    }
    
    public function updateCenter(Request $request, Center $center)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:centers,name,' . $center->id,
            'location' => 'required|string|max:255',
            'manager_id' => 'nullable|exists:partners,id',
            'is_active' => 'boolean',
        ]);

        // Remove current manager status
        Partner::where('center_id', $center->id)->where('is_manager', true)->update(['is_manager' => false]);
        
        $center->update([
            'name' => $validated['name'],
            'location' => $validated['location'],
            'is_active' => $validated['is_active'] ?? true,
        ]);
        
        if ($validated['manager_id']) {
            $partner = Partner::find($validated['manager_id']);
            $partner->update([
                'center_id' => $center->id,
                'is_manager' => true
            ]);
            
            $center->update([
                'manager_name' => $partner->name,
                'contact_email' => $partner->email,
                'contact_phone' => $partner->phone ?? '',
            ]);
        } else {
            $center->update([
                'manager_name' => '',
                'contact_email' => '',
                'contact_phone' => '',
            ]);
        }
        
        return redirect()->route('admin.centers')->with('success', 'Center updated successfully!');
    }
    
    // Enrollments Management
    public function enrollments(Request $request)
    {
        $query = StudentCourse::with(['student.center', 'course']);
        
        if ($request->course_id) {
            $query->where('course_id', $request->course_id);
        }
        
        if ($request->center_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('center_id', $request->center_id);
            });
        }
        
        // Create metrics query with same filters
        $metricsQuery = StudentCourse::with(['course']);
        
        if ($request->course_id) {
            $metricsQuery->where('course_id', $request->course_id);
        }
        
        if ($request->center_id) {
            $metricsQuery->whereHas('student', function($q) use ($request) {
                $q->where('center_id', $request->center_id);
            });
        }
        
        $metrics = [
            'total_enrollments' => (clone $metricsQuery)->count(),
            'total_income' => (clone $metricsQuery)->sum('amount_paid'),
            'outstanding_amount' => (clone $metricsQuery)->join('courses', 'student_courses.course_id', '=', 'courses.id')
                                                         ->selectRaw('SUM(CASE WHEN (courses.price - COALESCE(student_courses.amount_paid, 0)) > 0 THEN (courses.price - COALESCE(student_courses.amount_paid, 0)) ELSE 0 END) as outstanding')
                                                         ->value('outstanding') ?: 0,
            'issued_certificates' => (clone $metricsQuery)->where('is_certified', true)->count(),
            'due_certificates' => (clone $metricsQuery)->join('courses', 'student_courses.course_id', '=', 'courses.id')
                                                        ->whereRaw('CAST(COALESCE(student_courses.amount_paid, 0) AS DECIMAL(10,2)) >= CAST(courses.price AS DECIMAL(10,2))')
                                                        ->where('student_courses.is_certified', false)
                                                        ->count()
        ];
        
        $enrollments = $query->latest()->paginate(15)->withQueryString();
        $courses = Course::where('is_active', true)->get();
        $centers = Center::where('is_active', true)->get();
        
        return Inertia::render('Admin/Enrollments', [
            'enrollments' => $enrollments,
            'courses' => $courses,
            'centers' => $centers,
            'metrics' => $metrics,
            'filters' => $request->only(['course_id', 'center_id'])
        ]);
    }
    
    public function addPayment(Request $request, StudentCourse $enrollment)
    {
        $enrollment->load('course');
        
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'payment_date' => 'required|date',
        ]);

        // Validate that total payment doesn't exceed course price
        $newTotal = $enrollment->amount_paid + $validated['amount'];
        if ($newTotal > $enrollment->course->price) {
            $remaining = $enrollment->course->price - $enrollment->amount_paid;
            return back()->with('error', "Payment amount exceeds remaining balance. Maximum allowed: ₦{$remaining}");
        }

        $paymentRecord = PaymentRecord::create([
            'student_course_id' => $enrollment->id,
            'amount' => $validated['amount'],
            'payment_method' => $validated['payment_method'],
            'notes' => $validated['notes'],
            'payment_date' => $validated['payment_date'],
            'recorded_by_type' => 'admin',
            'recorded_by_id' => Auth::id(),
            'recorded_by_name' => Auth::user()->name,
        ]);

        $enrollment->amount_paid = $newTotal;
        $enrollment->save();

        ActivityLog::create([
            'student_id' => $enrollment->student_id,
            'action' => 'payment_added',
            'description' => "Payment of ₦{$validated['amount']} added",
            'performed_by_type' => 'admin',
            'performed_by_id' => Auth::id(),
            'performed_by_name' => Auth::user()->name,
            'new_values' => ['amount' => $validated['amount'], 'total_paid' => $enrollment->amount_paid],
        ]);

        return back()->with('success', 'Payment recorded successfully!');
    }
    
    public function viewRecords(StudentCourse $enrollment)
    {
        $enrollment->load(['student', 'course', 'paymentRecords']);
        
        $paymentRecords = $enrollment->paymentRecords()->latest()->get();
        $activityLogs = ActivityLog::where('student_id', $enrollment->student_id)->latest()->get();
        
        return Inertia::render('Admin/EnrollmentRecords', [
            'enrollment' => $enrollment,
            'paymentRecords' => $paymentRecords,
            'activityLogs' => $activityLogs
        ]);
    }
    
    public function issueCertificate(StudentCourse $enrollment)
    {
        $enrollment->load('course');
        
        if (!$enrollment->isFullyPaid()) {
            $remaining = $enrollment->remaining_amount;
            return back()->with('error', "Cannot issue certificate. Payment not complete. Remaining: ₦" . number_format($remaining, 2));
        }

        if ($enrollment->certificate_number) {
            return back()->with('error', 'Certificate already issued.');
        }
        
        // Check certificate limit if enabled
        if (config('certificates.limit.enabled')) {
            $totalCertificates = StudentCourse::where('is_certified', true)->count() + \App\Models\ManualCertificateRecord::count();
            $maxRecords = config('certificates.limit.max_records', 50);
            if ($totalCertificates >= $maxRecords) {
                return back()->with('error', "Certificate limit reached ({$maxRecords} records maximum). Cannot issue more certificates.");
            }
        }
        
        // Check if there are sufficient credits
        $remainingCredits = \App\Models\CertificateCredit::getTotalCreditsRemaining();
        if ($remainingCredits <= 0) {
            return back()->with('error', 'Insufficient certificate credits. Please purchase more credits to issue certificates.');
        }

        // Use one credit
        if (!\App\Models\CertificateCredit::useCredit()) {
            return back()->with('error', 'Failed to use certificate credit. Please try again.');
        }
        
        $enrollment->update([
            'certificate_number' => 'CERT-' . strtoupper(Str::random(8)),
            'is_certified' => true,
            'certification_date' => now(),
            'issued_by_type' => 'admin',
            'issued_by_id' => Auth::id(),
            'issued_by_name' => Auth::user()->name,
        ]);

        ActivityLog::create([
            'student_id' => $enrollment->student_id,
            'action' => 'certificate_issued',
            'description' => "Certificate {$enrollment->certificate_number} issued",
            'performed_by_type' => 'admin',
            'performed_by_id' => Auth::id(),
            'performed_by_name' => Auth::user()->name,
            'new_values' => ['certificate_number' => $enrollment->certificate_number],
        ]);

        $successMessage = 'Certificate issued successfully!';
        if (config('certificates.limit.show_warnings')) {
            $successMessage .= ' This record is permanent and cannot be removed.';
        }

        return back()->with('success', $successMessage);
    }
    
    public function certificates(Request $request)
    {
        $query = StudentCourse::with(['student.center', 'course'])
                             ->where('is_certified', true)
                             ->whereNotNull('certificate_number');
        
        if ($request->search) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('state_code', 'like', '%' . $request->search . '%');
            })->orWhere('certificate_number', 'like', '%' . $request->search . '%');
        }
        
        if ($request->course_id) {
            $query->where('course_id', $request->course_id);
        }
        
        if ($request->center_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('center_id', $request->center_id);
            });
        }
        
        $certificates = $query->latest('certification_date')->paginate(15)->withQueryString();
        $courses = Course::where('is_active', true)->get();
        $centers = Center::where('is_active', true)->get();
        
        return Inertia::render('Admin/Certificates', [
            'certificates' => $certificates,
            'courses' => $courses,
            'centers' => $centers,
            'filters' => $request->only(['search', 'course_id', 'center_id'])
        ]);
    }
    
    public function credits()
    {
        $credits = \App\Models\CertificateCredit::latest()->paginate(10);
        $totalCredits = \App\Models\CertificateCredit::getTotalCreditsRemaining();
        
        return Inertia::render('Admin/Credits', [
            'credits' => $credits,
            'totalCredits' => $totalCredits
        ]);
    }
    
    public function purchaseCredits(Request $request)
    {
        $validated = $request->validate([
            'credits_purchased' => 'required|integer|min:1|max:1000',
            'cost_per_credit' => 'required|numeric|min:0.01',
            'notes' => 'nullable|string|max:500'
        ]);
        
        $totalCost = $validated['credits_purchased'] * $validated['cost_per_credit'];
        
        \App\Models\CertificateCredit::create([
            'credits_purchased' => $validated['credits_purchased'],
            'credits_remaining' => $validated['credits_purchased'],
            'cost_per_credit' => $validated['cost_per_credit'],
            'total_cost' => $totalCost,
            'purchased_by_type' => 'admin',
            'purchased_by_id' => Auth::id(),
            'purchased_by_name' => Auth::user()->name,
            'notes' => $validated['notes']
        ]);
        
        return back()->with('success', "Successfully purchased {$validated['credits_purchased']} certificate credits for ₦" . number_format($totalCost, 2));
    }
    
    public function manualCertificates(Request $request)
    {
        $query = \App\Models\ManualCertificateRecord::with('center');
        
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('student_name', 'like', '%' . $request->search . '%')
                  ->orWhere('certificate_number', 'like', '%' . $request->search . '%')
                  ->orWhere('course_name', 'like', '%' . $request->search . '%');
            });
        }
        
        if ($request->center_id) {
            $query->where('center_id', $request->center_id);
        }
        
        $records = $query->latest()->paginate(15)->withQueryString();
        $centers = Center::where('is_active', true)->get();
        $courses = Course::where('is_active', true)->get();
        
        // Calculate certificate counts if limit is enabled
        $certificateData = [];
        if (config('certificates.limit.enabled') && config('certificates.limit.show_counter')) {
            $totalCertificates = StudentCourse::where('is_certified', true)->count() + \App\Models\ManualCertificateRecord::count();
            $maxRecords = config('certificates.limit.max_records', 50);
            $certificateData = [
                'totalCertificates' => $totalCertificates,
                'maxRecords' => $maxRecords,
                'remainingSlots' => max(0, $maxRecords - $totalCertificates),
                'showCounter' => true,
                'showWarnings' => config('certificates.limit.show_warnings', true)
            ];
        } else {
            $certificateData = [
                'showCounter' => false,
                'showWarnings' => config('certificates.limit.show_warnings', true)
            ];
        }
        
        return Inertia::render('Admin/ManualCertificates', array_merge([
            'records' => $records,
            'centers' => $centers,
            'courses' => $courses,
            'filters' => $request->only(['search', 'center_id'])
        ], $certificateData));
    }
    
    public function storeManualCertificate(Request $request)
    {
        // Check certificate limit if enabled
        if (config('certificates.limit.enabled')) {
            $totalCertificates = StudentCourse::where('is_certified', true)->count() + \App\Models\ManualCertificateRecord::count();
            $maxRecords = config('certificates.limit.max_records', 50);
            if ($totalCertificates >= $maxRecords) {
                return back()->with('error', "Certificate limit reached ({$maxRecords} records maximum). Cannot add more certificates.");
            }
        }
        
        $validated = $request->validate([
            'certificate_number' => 'required|string|unique:manual_certificate_records,certificate_number|max:255',
            'student_name' => 'required|string|max:255',
            'course_id' => 'required|exists:courses,id',
            'issuance_date' => 'required|date',
            'center_id' => 'nullable|exists:centers,id',
            'notes' => 'nullable|string|max:500'
        ]);
        
        $course = Course::find($validated['course_id']);
        
        \App\Models\ManualCertificateRecord::create([
            'certificate_number' => $validated['certificate_number'],
            'student_name' => $validated['student_name'],
            'course_name' => $course->name,
            'issuance_date' => $validated['issuance_date'],
            'center_id' => $validated['center_id'],
            'uploaded_by_type' => 'admin',
            'uploaded_by_id' => Auth::id(),
            'uploaded_by_name' => Auth::user()->name,
            'notes' => $validated['notes']
        ]);
        
        $successMessage = 'Manual certificate record added successfully!';
        if (config('certificates.limit.show_warnings')) {
            $successMessage .= ' This record is permanent and cannot be removed.';
        }
        
        return back()->with('success', $successMessage);
    }
    
    private function getTimeRange($timeFilter, $customMonth = null, $customYear = null)
    {
        $now = now();
        
        switch ($timeFilter) {
            case 'current_month':
                $start = $now->copy()->startOfMonth();
                $end = $now->copy()->endOfMonth();
                $label = 'This Month (' . $start->format('M Y') . ')';
                break;
                
            case 'last_month':
                $start = $now->copy()->subMonth()->startOfMonth();
                $end = $now->copy()->subMonth()->endOfMonth();
                $label = 'Last Month (' . $start->format('M Y') . ')';
                break;
                
            case 'current_year':
                $start = $now->copy()->startOfYear();
                $end = $now->copy()->endOfYear();
                $label = 'This Year (' . $start->format('Y') . ')';
                break;
                
            case 'last_year':
                $start = $now->copy()->subYear()->startOfYear();
                $end = $now->copy()->subYear()->endOfYear();
                $label = 'Last Year (' . $start->format('Y') . ')';
                break;
                
            case 'custom_month':
                if ($customMonth && $customYear) {
                    $start = now()->setYear($customYear)->setMonth($customMonth)->startOfMonth();
                    $end = now()->setYear($customYear)->setMonth($customMonth)->endOfMonth();
                    $label = $start->format('M Y');
                } else {
                    // Fallback to current month
                    $start = $now->copy()->startOfMonth();
                    $end = $now->copy()->endOfMonth();
                    $label = 'This Month (' . $start->format('M Y') . ')';
                }
                break;
                
            case 'all_time':
            default:
                $start = now()->subYears(10); // Go back 10 years
                $end = now()->addYear(); // Go forward 1 year
                $label = 'All Time';
                break;
        }
        
        return [$start, $end, $label];
    }
}
